<?php

namespace App\DataTables;

use App\Models\Inventory;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Gate;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class InventoryDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
        ->addIndexColumn()
        ->editColumn('entry_date',function($inventory){
            return $inventory->entry_date ? $inventory->entry_date->format('d-m-Y') : "";
        })
        ->editColumn('product.name',function($inventory){
            $product = $inventory->product;
            return $product ? $product->name : '';
        })
        ->editColumn('quantity', function ($inventory) {
            return  (int)$inventory->quantity ?? '';
        })
        ->editColumn('place', function ($inventory) {
            return config('app.product_place.'.$inventory->place) ?? '';
        })
        ->editColumn('type', function ($inventory) {
            return config('app.type.' . $inventory->type) ?? '';
        })

        ->addColumn('action',function($inventory){
            $action='<div>';
            if (Gate::check('inventory_access')) {
                $viewIcon = view('components.svg-icon', ['icon' => 'view'])->render();
                $action .= '<button title="'.trans('quickadmin.qa_view').'" class="btn btn-icon btn-info view-inventory-btn p-1" data-href="'.route('inventory.show', $inventory->uuid).'">'.$viewIcon.'</button>';
            }
            if (Gate::check('inventory_edit')) {
                $editIcon = view('components.svg-icon', ['icon' => 'edit'])->render();
                $action .= '<button title="'.trans('quickadmin.qa_edit').'" class="btn btn-icon btn-info edit-inventory-btn p-1" data-href="'.route('inventory.edit', $inventory->uuid).'">'.$editIcon.'</button>';
            }
            
            // if (Gate::check('inventory_delete')) {                
            //    $deleteIcon = view('components.svg-icon', ['icon' => 'delete'])->render();
            //    $action .= '<button title="'.trans('quickadmin.qa_delete').'" class="btn btn-icon btn-info delete-inventory-btn p-1" data-href="'.route('inventory.destroy', $inventory->uuid).'">'.$deleteIcon.'</button>';                
            // }
            $action .='</div>';
            return $action;
        })
        ->filterColumn('entry_date', function ($query, $keyword) {
            $query->whereRaw("DATE_FORMAT(inventories.entry_date,'%d-%m-%Y') like ?", ["%$keyword%"]); //date_format when searching using date
        })

        ->rawColumns(['action']);
            
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Inventory $model): QueryBuilder
    {
        return $model->newQuery()->with(['product']);
    }
    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    { 
        return $this->builder()
        ->setTableId('inventory-table')
        ->parameters([
            'responsive' => false,
            'pageLength' => 50,
        ])
        ->columns($this->getColumns())
        ->minifiedAjax()
        ->dom('frtip')
        ->orderBy(1);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('DT_RowIndex')->title(trans('quickadmin.qa_sn'))->orderable(false)->searchable(false),
            Column::make('entry_date')->title(trans('quickadmin.inventory.fields.entry_date')),
            Column::make('product.name')->title(trans('quickadmin.inventory.fields.product_name')),
           // Column::make('area.address')->title(trans('quickadmin.inventory.fields.area_id')),
            Column::make('place')->title(trans('quickadmin.inventory.fields.place')),
            Column::make('quantity')->title(trans('quickadmin.inventory.fields.quantity')),
            Column::make('type')->title(trans('quickadmin.inventory.fields.type')),
            
            Column::computed('action')
            ->exportable(false)
            ->printable(false)
            ->width(60)
            ->addClass('text-center')->title(trans('quickadmin.qa_action')),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Inventory_' . date('YmdHis');
    }
}
