<?php

namespace App\Http\Controllers\Admin;

use App\DataTables\CashManagementDataTable;
use App\Http\Controllers\Controller;
use App\Http\Requests\CashManagement\StoreUpdateCashManagementRequest;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\CashManagement;
use Carbon\Carbon;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Gate;

class CashManagementController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(CashManagementDataTable $dataTable)
    {
        abort_if(Gate::denies('cash_management_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        return $dataTable->render('admin.cash_management.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
       abort_if(Gate::denies('cash_management_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $customers = User::where('is_active',1)->where('id','<>',1)->orderBy('name')->get();
        $invoice_number = getNewInvoiceNumber('','new_cash_staff_manage',Carbon::now()->format('Y-m-d'));
        return view('admin.cash_management.create', compact('customers','invoice_number'));
    }

    /**
     * Store a newly created resource in storage.
     */
    // public function store(Request $request)
    public function store(StoreUpdateCashManagementRequest $request)
    {
         abort_if(Gate::denies('cash_management_create'), Response::HTTP_FORBIDDEN, '403 Forbidden');
         $paymentTransaction = CashManagement::create([
            'user_id'       => $request->user_id,
            'voucher_number'    => $request->voucher_number ? $request->voucher_number : getNewInvoiceNumber('','new_cash_staff_manage',$request->entry_date),
            'payment_way'       => $request->payment_way,            
            'amount'            => $request->amount,
            'entry_date'        => $request->entry_date,
            'remark'            => $request->remark ?? '',
        ]);

        if($request->payment_way == 'by_cash'){
            storeCashBalanceRecord($paymentTransaction, 'debit',$request->entry_date);
        }

        addToLog($request,'Cash Management','Create', $paymentTransaction);

        return response()->json([
            'success' => true,
            'message' => trans('messages.crud.add_record'),
            'alert-type'=> trans('quickadmin.alert-type.success'),
            'invoiceNumber'     => 'Cash Receipt No. #'.$paymentTransaction->voucher_number,
            'redirectUrl' => route('cash-management.index'),
        ], 200);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request,CashManagement $cashManagement)
    {
        abort_if(Gate::denies('cash_management_access'), Response::HTTP_FORBIDDEN, '403 Forbidden');
        $htmlView = view('admin.cash_management.show',compact('cashManagement'))->render();
        return response()->json(['success' => true, 'htmlView' => $htmlView]);        
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(CashManagement $cashManagement)
    {
        abort_if(Gate::denies('cash_management_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');  
        $customers = User::where('is_active',1)->where('id','<>',1)->orderBy('name')->get();
        return view('admin.cash_management.edit', compact('customers', 'cashManagement'));

    }

    /**
     * Update the specified resource in storage.
     */
    public function update(StoreUpdateCashManagementRequest $request, CashManagement $cashManagement)
    {
       abort_if(Gate::denies('cash_management_edit'), Response::HTTP_FORBIDDEN, '403 Forbidden');            
        $oldvalue = $cashManagement->getOriginal();
        $cashManagement->update($request->all());
        $newValue = $cashManagement->refresh();
        addToLog($request,'Cash Management','Edit', $newValue ,$oldvalue);  
        
        return response()->json([
            'success' => true,
            'message' => trans('messages.crud.update_record'),
            'alert-type'=> trans('quickadmin.alert-type.success'),
            'invoiceNumber'     => 'Cash Receipt No. #'.$cashManagement->voucher_number,
            'redirectUrl' => route('cash-management.index'),
        ], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(CashManagement $cashManagement)
    {
       abort_if(Gate::denies('cash_management_delete'), Response::HTTP_FORBIDDEN, '403 Forbidden');      
        $cashManagement->delete();
        return response()->json(['success' => true,
         'message' => trans('messages.crud.delete_record'),
         'alert-type'=> trans('quickadmin.alert-type.success'),
         'title' => trans('quickadmin.cash-management.title')
        ], 200);
    }
}
