<?php

namespace App\Models;

use App\Traits\UuidTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PaymentTransaction extends Model
{
    use HasFactory, SoftDeletes, UuidTrait;

    public $timestamps = true;

    protected $casts = [
        'entry_date' => 'date',
    ];

    protected $fillable = [
        'uuid',
        'customer_id',
        'payment_type',
        'payment_way',
        'voucher_number',
        'order_id',
        'extra_details',
        'remark',
        'generated_from',
        'amount',
        'entry_date', 
        'created_by',
        'updated_by',
        'deleted_by',        
        'is_modified',
        'deleted_at',
    ];

    public function getRouteKeyName()
    {
        return 'uuid';
    }

    protected static function boot()
    {
        parent::boot();
        static::creating(function(PaymentTransaction $model) {
            $model->created_by = auth()->user()->id;
        });

        static::deleting(function(PaymentTransaction $model) {
            $model->deleted_by = auth()->user()->id;
            $model->save();
        });

        static::updating(function(PaymentTransaction $model) {
            $model->updated_by = auth()->user()->id;
        });
    }

    public function user(){
        return $this->belongsTo(User::class, 'created_by');
    }

    public function deletedByUser(){
        return $this->belongsTo(User::class, 'deleted_by');
    }

    public function customer(){
        return $this->belongsTo(Customer::class, 'customer_id');
    }

    public function order(){
        return $this->belongsTo(Order::class, 'order_id');
    }

    public function cashReceiptAmounts()
    {
        return $this->hasMany(CashReceiptAmount::class);
    }

    /**
     * Polymorphic relationship with CashBalanceRecord.
     */
    public function cashBalanceRecords()
    {
        return $this->morphMany(CashBalanceRecord::class, 'recordable');
    }
}
