import { Product } from '@/types/product';

interface ApiResponse {
  status: boolean;
  message: string;
  data: {
    data: Product[];
    pagination: {
      totalItems: number;
      totalPages: number;
      currentPage: number;
      pageSize: number;
    };
  };
}

const API_BASE_URL = process.env.NEXT_PUBLIC_API_BASE_URL || 'http://localhost:3000/api/v1';


export async function fetchProducts(page?: number, limit?: number, category_type? : string, category? : string, sub_category? : string): Promise<{
    products: Product[];
    pagination: {
      totalItems: number;
      totalPages: number;
      currentPage: number;
      pageSize: number;
    };
  }> {

    try {
      const queryParams = new URLSearchParams();
      if (page !== undefined) queryParams.append('page', String(page));
      if (limit !== undefined) queryParams.append('limit', String(limit));
      if (category_type !== undefined) queryParams.append('category_type', String(category_type ));
      if (category !== undefined) queryParams.append('category', String(category));
      if (sub_category !== undefined) queryParams.append('sub_category', String(sub_category ));

      const res = await fetch(`${API_BASE_URL}/products?${queryParams.toString()}`);
      
      if (!res.ok) {
          throw new Error('Failed to fetch products');
      }

      const result: ApiResponse = await res.json();
      
      return {
          products: result.data.data,
          pagination: result.data.pagination,
      };
    } catch (error) {
      console.error('Product fetch error:', error);

      // Return empty fallback for safe SSG
      return {
        products: [],
        pagination: {
          totalItems: 0,
          totalPages: 0,
          currentPage: 1,
          pageSize: 10,
        },
      };
    }
}
