# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Development Commands

```bash
# Development
npm run dev          # Start custom server with WebSocket support on port 3005
npm run dev:next     # Start standard Next.js dev server (no WebSocket)
npm run build        # Build for production  
npm run start        # Start production server with WebSocket support
npm run start:next   # Start standard Next.js production server (no WebSocket)
npm run lint         # Run ESLint

# Database operations
npm run db:push      # Push Prisma schema to database
npm run db:generate  # Generate Prisma client
npm run db:seed      # Seed database with sample data
npm run db:studio    # Open Prisma Studio

# PM2 process management (production)
npm run pm2:dev      # Start with PM2 in dev mode
npm run pm2:start    # Start with PM2 in production
npm run pm2:stop     # Stop PM2 process
npm run pm2:delete   # Delete PM2 process
npm run pm2:status   # Check PM2 status
```

## Architecture Overview

### Multi-Country Platform
This is a freelancer marketplace supporting 10 European countries with:
- Country-specific domains (smile24.lu, smile24.fr, etc.)
- Localized pricing (€9.90 for most countries, CHF 19.90 for Switzerland)
- Multi-language support (6 languages: en, fr, de, it, pt, es)
- Currency localization per country

### Tech Stack
- **Framework**: Next.js 15 with App Router and Turbopack
- **Database**: MySQL with Prisma ORM
- **Authentication**: NextAuth.js with social login (Google, Facebook)
- **Styling**: Bootstrap 5 + custom CSS
- **Forms**: React Hook Form + Zod validation
- **Payments**: Stripe (+ Twint for Switzerland)
- **Real-time**: Custom WebSocket server for instant messaging

### User Roles & Flow
Three main user types with distinct workflows:
- **ADMIN**: Platform management
- **CLIENT**: Post projects, hire freelancers, manage work
- **FREELANCER**: Bid on projects, complete work, requires subscription

## Key Architecture Patterns

### Route Structure
```
src/app/[locale]/
├── (auth)/          # Authentication pages (login, signup)
├── (dashboard)/     # Protected user dashboards
│   ├── client/      # Client-specific pages
│   ├── freelancer/  # Freelancer-specific pages with role-based project views
│   │   ├── projects/[projectId]/        # Generic project view
│   │   ├── projects/ongoing/[projectId] # Ongoing project details
│   │   ├── projects/completed/[projectId] # Completed project details
│   │   └── projects/cancelled/[projectId] # Cancelled project details
│   └── layout.tsx   # Shared dashboard layout
├── (public)/        # Public pages (homepage, categories, etc.)
└── freelancer/      # Legacy freelancer routes (being migrated)
```

### Database Schema
Core entities in Prisma schema:
- **User**: Authentication and basic info
- **Profile**: Extended user details
- **Project**: Client job postings with status tracking
- **Bid**: Freelancer proposals with status management
- **Subscription**: Freelancer subscription management
- **Message**: Real-time communication system
- **Review**: Rating and feedback system

### Middleware Logic
- Country detection by domain with fallback to localhost config
- Locale routing with country-specific language defaults
- Role-based route protection for dashboard areas
- Session management with JWT strategy

### API Architecture
RESTful API routes organized by feature:
- `/api/auth/*` - Authentication endpoints
- `/api/freelancer/*` - Freelancer-specific operations (memberships, projects, messages)
- `/api/projects/*` - Project management with progress tracking and bidding
- `/api/messages/*` - Real-time messaging with conversations and WebSocket chat
- `/api/notifications/*` - Complete notification system (mark read, unread count, CRUD)
- `/api/reviews/*` - Rating and review system
- `/api/ws/*` - WebSocket endpoints for real-time features (project chat)
- `/api/diagnostic/*` - System health and diagnostic tools

## Development Guidelines

### File Organization
- Components are feature-organized under `/components/[feature]/`:
  - `auth/` - Authentication components (RememberMeStatus)
  - `chat/` - Real-time messaging (ProjectChat)  
  - `notifications/` - Notification system (NotificationBell, NotificationList, NotificationPanel)
  - `reviews/` - Review and rating components
  - `ui/` - Rich text editing components (RichTextarea, SimpleTextEditor)
  - `messages/` - Communication components
- API routes mirror the frontend route structure
- Types are centralized in `/types/` (bid.ts, project-details.ts, etc.)
- Utilities in `/lib/` for shared logic

### State Management
- Server state via Prisma + MySQL
- Client state through React Hook Form for forms
- Session state via NextAuth.js with remember me functionality
- Real-time updates via WebSocket connections (project chat, notifications)
- Rich text editing state managed by custom text editor components

### Authentication Flow
- JWT-based sessions with 30-day expiration
- Role-based access control in middleware
- Protected routes redirect to login with return URL
- Remember me functionality extends session duration

### Styling Approach
- Bootstrap 5 for component library
- Custom CSS modules for specific components
- Responsive design with mobile-first approach
- Consistent color scheme and typography

## Testing & Quality

### Sample Users (after seeding)
- Admin: admin@smile24.com / admin123
- Client: client@example.com / admin123
- Freelancer: freelancer@example.com / admin123

## Environment Configuration

Required environment variables:
```env
# Database
DATABASE_URL="mysql://user:password@localhost:3306/smile24"

# NextAuth
NEXTAUTH_URL="http://localhost:3005"
NEXTAUTH_SECRET="your-secret-key"

# OAuth Providers
GOOGLE_CLIENT_ID="your-google-client-id"
GOOGLE_CLIENT_SECRET="your-google-client-secret"
FACEBOOK_CLIENT_ID="your-facebook-client-id"
FACEBOOK_CLIENT_SECRET="your-facebook-client-secret"

# Stripe Payment
STRIPE_PUBLISHABLE_KEY="pk_test_..."
STRIPE_SECRET_KEY="sk_test_..."
STRIPE_WEBHOOK_SECRET="whsec_..."

# Application
NEXT_PUBLIC_APP_URL="http://localhost:3005"

# Email (if using Nodemailer)
EMAIL_SERVER_HOST="smtp.example.com"
EMAIL_SERVER_PORT=587
EMAIL_SERVER_USER="your-email@example.com"
EMAIL_SERVER_PASSWORD="your-password"
EMAIL_FROM="noreply@smile24.com"
```

## Important Notes

- Default development port is 3005 (not 3000)
- All freelancers require active subscriptions to bid on projects
- **WebSocket Real-time Features**:
  - Instant messaging between users
  - Project-specific chat rooms
  - Bid-specific chat threads
  - Real-time typing indicators
  - Automatic fallback to polling if WebSocket unavailable
  - WebSocket server runs on port 3005 alongside Next.js
- File uploads for profiles and project documents stored in `/public/uploads/`
- Country configuration drives pricing and payment methods
- Turbopack is enabled for faster development builds
- Rich text editing capabilities available for project descriptions and messages
- Complete notification system with real-time updates
- Remember me functionality extends session duration beyond standard 30 days
- Role-specific project views (ongoing, completed, cancelled) for freelancers
- Comprehensive review and rating system between clients and freelancers

## Testing & Quality

Currently no testing framework is configured. Consider adding Jest, Vitest, or Playwright for comprehensive testing coverage.