'use client';

import Link from 'next/link';
import { useState } from 'react';
import { Card, Row, Col, Badge, Button, Table } from 'react-bootstrap';

interface ClientDashboardProps {
  user: any;
  stats: {
    totalProjects: number;
    openProjects: number;
    inProgressProjects: number;
    completedProjects: number;
    totalBids: number;
  };
  locale: string;
}

export default function ClientDashboard({ user, stats, locale }: ClientDashboardProps) {
  const [activeTab, setActiveTab] = useState('overview');

  const getStatusBadge = (status: string) => {
    const statusConfig = {
      DRAFT: { variant: 'secondary', text: 'Draft' },
      OPEN: { variant: 'success', text: 'Open' },
      IN_PROGRESS: { variant: 'warning', text: 'In Progress' },
      COMPLETED: { variant: 'primary', text: 'Completed' },
      CANCELLED: { variant: 'danger', text: 'Cancelled' }
    };
    
    const config = statusConfig[status as keyof typeof statusConfig] || { variant: 'secondary', text: status };
    return <Badge bg={config.variant}>{config.text}</Badge>;
  };

  const formatDate = (date: string) => {
    return new Date(date).toLocaleDateString();
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-US', {
      style: 'currency',
      currency: 'EUR'
    }).format(amount);
  };

  return (
    <div className="dashboard-page">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashboard-header">
              <h1>Client Dashboard</h1>
              <p>Welcome back, {user.profile?.firstName || user.email}!</p>
            </div>
          </div>
        </div>

        {/* Stats Cards */}
        <Row className="mb-4">
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.totalProjects}</div>
                  <div className="stat-label">Total Projects</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.openProjects}</div>
                  <div className="stat-label">Open Projects</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.inProgressProjects}</div>
                  <div className="stat-label">In Progress</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
          <Col md={3}>
            <Card className="stat-card">
              <Card.Body>
                <div className="stat-content">
                  <div className="stat-number">{stats.totalBids}</div>
                  <div className="stat-label">Total Bids</div>
                </div>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {/* Quick Actions */}
        <Row className="mb-4">
          <Col>
            <Card>
              <Card.Body>
                <h5>Quick Actions</h5>
                <div className="d-flex gap-2 flex-wrap">
                  <Link href={`/${locale}/client/post-project`}>
                    <Button variant="primary">Post New Project</Button>
                  </Link>
                  <Link href={`/${locale}/client/projects`}>
                    <Button variant="outline-primary">View All Projects</Button>
                  </Link>
                  <Link href={`/${locale}/client/profile`}>
                    <Button variant="outline-secondary">Edit Profile</Button>
                  </Link>
                </div>
              </Card.Body>
            </Card>
          </Col>
        </Row>

        {/* Recent Projects */}
        <Row>
          <Col>
            <Card>
              <Card.Header>
                <h5 className="mb-0">Recent Projects</h5>
              </Card.Header>
              <Card.Body>
                {user.clientProjects.length > 0 ? (
                  <Table responsive>
                    <thead>
                      <tr>
                        <th>Project</th>
                        <th>Category</th>
                        <th>Budget</th>
                        <th>Status</th>
                        <th>Bids</th>
                        <th>Created</th>
                        <th>Actions</th>
                      </tr>
                    </thead>
                    <tbody>
                      {user.clientProjects.slice(0, 5).map((project: any) => (
                        <tr key={project.id}>
                          <td>
                            <div>
                              <strong>{project.title}</strong>
                              <br />
                              <small className="text-muted">
                                {project.description.substring(0, 50)}...
                              </small>
                            </div>
                          </td>
                          <td>{project.category.name}</td>
                          <td>{formatCurrency(project.budget)}</td>
                          <td>{getStatusBadge(project.status)}</td>
                          <td>{project.bids.length}</td>
                          <td>{formatDate(project.createdAt)}</td>
                          <td>
                            <Link href={`/${locale}/client/projects/${project.id}`}>
                              <Button size="sm" variant="outline-primary">View</Button>
                            </Link>
                          </td>
                        </tr>
                      ))}
                    </tbody>
                  </Table>
                ) : (
                  <div className="text-center py-4">
                    <p className="text-muted">No projects yet</p>
                    <Link href={`/${locale}/client/post-project`}>
                      <Button variant="primary">Post Your First Project</Button>
                    </Link>
                  </div>
                )}
              </Card.Body>
            </Card>
          </Col>
        </Row>
      </div>
    </div>
  );
} 