'use client';

import { useState } from 'react';

interface StarRatingProps {
  rating: number;
  onRatingChange: (rating: number) => void;
  readonly?: boolean;
  size?: 'sm' | 'md' | 'lg';
  showText?: boolean;
}

const ratingTexts = {
  1: 'Poor',
  2: 'Fair', 
  3: 'Good',
  4: 'Very Good',
  5: 'Excellent'
};

export default function StarRating({ 
  rating, 
  onRatingChange, 
  readonly = false,
  size = 'md',
  showText = false 
}: StarRatingProps) {
  const [hoverRating, setHoverRating] = useState(0);

  const getSizeProps = () => {
    switch (size) {
      case 'sm':
        return { width: 16, height: 16 };
      case 'lg':
        return { width: 24, height: 24 };
      default:
        return { width: 20, height: 20 };
    }
  };

  const sizeProps = getSizeProps();
  const displayRating = readonly ? rating : (hoverRating || rating);

  const handleClick = (starRating: number) => {
    if (!readonly) {
      onRatingChange(starRating);
    }
  };

  const handleMouseEnter = (starRating: number) => {
    if (!readonly) {
      setHoverRating(starRating);
    }
  };

  const handleMouseLeave = () => {
    if (!readonly) {
      setHoverRating(0);
    }
  };

  return (
    <div className={`star-rating ${readonly ? 'readonly' : 'interactive'}`}>
      <div className="stars-container">
        {[1, 2, 3, 4, 5].map((star) => (
          <button
            key={star}
            type="button"
            className={`star ${star <= displayRating ? 'filled' : 'empty'}`}
            onClick={() => handleClick(star)}
            onMouseEnter={() => handleMouseEnter(star)}
            onMouseLeave={handleMouseLeave}
            disabled={readonly}
            aria-label={`${star} star${star !== 1 ? 's' : ''}`}
          >
            <svg
              width={sizeProps.width}
              height={sizeProps.height}
              viewBox="0 0 24 24"
              fill={star <= displayRating ? "#FB6107" : "none"}
              stroke="#FB6107"
              strokeWidth="1"
            >
              <path d="M12 .587l3.668 7.571 8.332 1.151-6.064 5.845 1.528 8.263L12 18.896l-7.464 4.521 1.528-8.263L.001 9.309l8.332-1.151z"/>
            </svg>
          </button>
        ))}
      </div>
      
      {showText && displayRating > 0 && (
        <span className="rating-text">
          {ratingTexts[displayRating as keyof typeof ratingTexts]}
        </span>
      )}

      <style jsx>{`
        .star-rating {
          display: flex;
          align-items: center;
          gap: 8px;
        }

        .stars-container {
          display: flex;
          gap: 2px;
        }

        .star {
          background: none;
          border: none;
          padding: 0;
          cursor: pointer;
          transition: all 0.2s ease;
          display: flex;
          align-items: center;
          justify-content: center;
        }

        .star:disabled {
          cursor: default;
        }

        .interactive .star:hover {
          transform: scale(1.1);
        }

        .interactive .star:active {
          transform: scale(0.95);
        }

        .rating-text {
          font-size: var(--font-small);
          font-weight: 500;
          color: var(--typo-secondary);
          margin-left: 4px;
        }

        @media (max-width: 768px) {
          .stars-container {
            gap: 1px;
          }
          
          .rating-text {
            font-size: var(--font-xs);
          }
        }
      `}</style>
    </div>
  );
}