import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function main() {
  console.log('🌱 Starting database seed...');

  // Clear existing data
  await prisma.notification.deleteMany();
  await prisma.review.deleteMany();
  await prisma.message.deleteMany();
  await prisma.bid.deleteMany();
  await prisma.project.deleteMany();
  await prisma.subscription.deleteMany();
  await prisma.payment.deleteMany();
  await prisma.package.deleteMany();
  await prisma.categoryProfile.deleteMany();
  await prisma.portfolio.deleteMany();
  await prisma.education.deleteMany();
  await prisma.experience.deleteMany();
  await prisma.categorySubscription.deleteMany();
  await prisma.profile.deleteMany();
  await prisma.user.deleteMany();
  await prisma.category.deleteMany();
  await prisma.language.deleteMany();
  await prisma.country.deleteMany();
  await prisma.siteSettings.deleteMany();

  console.log('🗑️  Cleared existing data');

  // Create languages
  const languages = await Promise.all([
    prisma.language.create({
      data: { code: 'en', name: 'English', nativeName: 'English', isActive: true }
    }),
    prisma.language.create({
      data: { code: 'fr', name: 'French', nativeName: 'Français', isActive: true }
    }),
    prisma.language.create({
      data: { code: 'de', name: 'German', nativeName: 'Deutsch', isActive: true }
    }),
    prisma.language.create({
      data: { code: 'it', name: 'Italian', nativeName: 'Italiano', isActive: true }
    }),
    prisma.language.create({
      data: { code: 'pt', name: 'Portuguese', nativeName: 'Português', isActive: true }
    }),
    prisma.language.create({
      data: { code: 'es', name: 'Spanish', nativeName: 'Español', isActive: true }
    })
  ]);

  console.log('🌍 Created languages');

  // Create countries
  const countries = await Promise.all([
    prisma.country.create({
      data: {
        id: 'luxembourg',
        name: 'Luxembourg',
        code: 'LU',
        domain: 'smile24.lu',
        languages: JSON.stringify(['fr', 'en']),
        defaultLanguage: 'fr',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'belgium',
        name: 'Belgium',
        code: 'BE',
        domain: 'smile24.be',
        languages: JSON.stringify(['fr', 'en']),
        defaultLanguage: 'fr',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'france',
        name: 'France',
        code: 'FR',
        domain: 'smile24.fr',
        languages: JSON.stringify(['fr', 'en']),
        defaultLanguage: 'fr',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'switzerland',
        name: 'Switzerland',
        code: 'CH',
        domain: 'smile24.ch',
        languages: JSON.stringify(['fr', 'de', 'it']),
        defaultLanguage: 'fr',
        currency: 'CHF',
        currencySymbol: 'CHF',
        packagePrice: 19.90,
        paymentGateways: JSON.stringify(['stripe', 'twint']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'italy',
        name: 'Italy',
        code: 'IT',
        domain: 'smile24.it',
        languages: JSON.stringify(['it', 'en']),
        defaultLanguage: 'it',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'uk',
        name: 'United Kingdom',
        code: 'GB',
        domain: 'smile24.uk',
        languages: JSON.stringify(['en']),
        defaultLanguage: 'en',
        currency: 'GBP',
        currencySymbol: '£',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'portugal',
        name: 'Portugal',
        code: 'PT',
        domain: 'smile24.pt',
        languages: JSON.stringify(['pt', 'en']),
        defaultLanguage: 'pt',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'spain',
        name: 'Spain',
        code: 'ES',
        domain: 'smile24.es',
        languages: JSON.stringify(['es', 'en']),
        defaultLanguage: 'es',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'germany',
        name: 'Germany',
        code: 'DE',
        domain: 'smile77.de',
        languages: JSON.stringify(['de', 'en']),
        defaultLanguage: 'de',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    }),
    prisma.country.create({
      data: {
        id: 'austria',
        name: 'Austria',
        code: 'AT',
        domain: 'smile77.at',
        languages: JSON.stringify(['de', 'en']),
        defaultLanguage: 'de',
        currency: 'EUR',
        currencySymbol: '€',
        packagePrice: 9.90,
        paymentGateways: JSON.stringify(['stripe']),
        isActive: true
      }
    })
  ]);

  console.log('🏳️  Created countries');

  // Create categories
  const categories = await Promise.all([
    prisma.category.create({
      data: {
        name: 'Agriculture',
        slug: 'agriculture',
        description: 'Agricultural services and farming solutions',
        icon: 'Agriculture.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Jewelry',
        slug: 'jewelry',
        description: 'Jewelry design, repair and crafting services',
        icon: 'Jewelry.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Bakery-Patisserie',
        slug: 'bakery-patisserie',
        description: 'Bakery and pastry services',
        icon: 'Bakery-Patisserie.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Tile',
        slug: 'tile',
        description: 'Tile installation and repair services',
        icon: 'Tile.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Bodywork',
        slug: 'bodywork',
        description: 'Automotive bodywork and repair services',
        icon: 'Bodywork.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Ceramic',
        slug: 'ceramic',
        description: 'Ceramic work and pottery services',
        icon: 'Ceramic.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Carpentry',
        slug: 'carpentry',
        description: 'Wood working and carpentry services',
        icon: 'Carpentry.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Heating',
        slug: 'heating',
        description: 'Heating system installation and maintenance',
        icon: 'Heating.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Metal construction',
        slug: 'metal-construction',
        description: 'Metal fabrication and construction services',
        icon: 'Metal-Construction.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Coronaries',
        slug: 'coronaries',
        description: 'Coronary and medical services',
        icon: 'Coronaries.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Cuisine',
        slug: 'cuisine',
        description: 'Cooking and culinary services',
        icon: 'Cuisine.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Cabinetmaking / Carpentry',
        slug: 'cabinetmaking-carpentry',
        description: 'Custom cabinetmaking and fine carpentry',
        icon: 'Cabinetmaking-Carpentry.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Scaffolding',
        slug: 'scaffolding',
        description: 'Scaffolding installation and rental services',
        icon: 'Scaffolding.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Electrical',
        slug: 'electrical',
        description: 'Electrical installation and repair services',
        icon: 'Electrical.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Electronics',
        slug: 'electronics',
        description: 'Electronic repair and installation services',
        icon: 'Electronics.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Forging',
        slug: 'forging',
        description: 'Metal forging and blacksmith services',
        icon: 'Forging.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Tightness',
        slug: 'tightness',
        description: 'Sealing and waterproofing services',
        icon: 'Tightness.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Forest and Garden',
        slug: 'forest-and-garden',
        description: 'Landscaping and garden maintenance services',
        icon: 'Forest-and-Garden.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Graphics',
        slug: 'graphics',
        description: 'Graphic design and visual communication services',
        icon: 'Graphics.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Watchmaking',
        slug: 'watchmaking',
        description: 'Watch repair and maintenance services',
        icon: 'Watchmaking.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Freemasonry',
        slug: 'freemasonry',
        description: 'Masonry and stonework services',
        icon: 'Freemasonry.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Auto Mechanic',
        slug: 'auto-mechanic',
        description: 'Automotive repair and maintenance services',
        icon: 'Auto-Mechanic.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Motorcycle Mechanics',
        slug: 'motorcycle-mechanics',
        description: 'Motorcycle repair and maintenance services',
        icon: 'Motorcycle-Mechanics.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Paint',
        slug: 'paint',
        description: 'Painting and decorating services',
        icon: 'Paint.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Goldsmith',
        slug: 'goldsmith',
        description: 'Jewelry making and goldsmith services',
        icon: 'Goldsmith.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Plastering',
        slug: 'plastering',
        description: 'Wall plastering and finishing services',
        icon: 'Plastering.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Ramonage',
        slug: 'ramonage',
        description: 'Chimney cleaning and maintenance services',
        icon: 'Ramonage.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Director/publicity trick',
        slug: 'director-publicity-trick',
        description: 'Media production and publicity services',
        icon: 'Director-Publicity-trick.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Sanitary',
        slug: 'sanitary',
        description: 'Plumbing and sanitary installation services',
        icon: 'Sanitary.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Sculpture',
        slug: 'sculpture',
        description: 'Sculpture and artistic carving services',
        icon: 'Sculpture.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Care',
        slug: 'care',
        description: 'Healthcare and personal care services',
        icon: 'Care.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Solar',
        slug: 'solar',
        description: 'Solar panel installation and maintenance',
        icon: 'solar.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Tapestry',
        slug: 'tapestry',
        description: 'Textile and tapestry services',
        icon: 'Tapestry.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Transports',
        slug: 'transports',
        description: 'Transportation and logistics services',
        icon: 'Transports.svg',
        isActive: true
      }
    }),
    prisma.category.create({
      data: {
        name: 'Ventilation',
        slug: 'ventilation',
        description: 'Ventilation system installation and maintenance',
        icon: 'Ventilation.svg',
        isActive: true
      }
    })
  ]);

  console.log('📂 Created categories');

  // Create packages for each country
  for (const country of countries) {
    await prisma.package.create({
      data: {
        name: 'Standard Package',
        description: 'Access to all categories, unlimited bids',
        features: JSON.stringify([
          'Access to all project categories',
          'Unlimited project bids',
          'Direct messaging with clients',
          'Profile visibility',
          '30-day subscription'
        ]),
        countryId: country.id,
        price: country.packagePrice,
        duration: 30,
        categoryLimit: null,
        bidLimit: null,
        isActive: true
      }
    });
  }

  console.log('📦 Created packages');

  // Create sample users
  const sampleUsers = await Promise.all([
    // Admin user
    prisma.user.create({
      data: {
        email: 'admin@smile24.com',
        password: '$2b$12$ihpo1gTiqrWZTfnemy9MzOdUqwOZb2cxRtKbsS4Yc84Q3lK0dyJcG', // password: admin@123
        role: 'ADMIN',
        provider: 'email',
        profile: {
          create: {
            firstName: 'Admin',
            lastName: 'User',
            countryId: 'switzerland',
            phone: '+352 123 456 789'
          }
        }
      }
    }),
    // Sample client
    prisma.user.create({
      data: {
        email: 'client@example.com',
        password: '$2b$12$ihpo1gTiqrWZTfnemy9MzOdUqwOZb2cxRtKbsS4Yc84Q3lK0dyJcG', // password: admin@123
        role: 'CLIENT',
        provider: 'email',
        profile: {
          create: {
            firstName: 'John',
            lastName: 'Client',
            countryId: 'switzerland',
            phone: '+352 987 654 321',
            bio: 'Looking for talented freelancers for various projects'
          }
        }
      }
    }),
    // Sample freelancer
    prisma.user.create({
      data: {
        email: 'freelancer@example.com',
        password: '$2b$12$ihpo1gTiqrWZTfnemy9MzOdUqwOZb2cxRtKbsS4Yc84Q3lK0dyJcG', // password: admin@123
        role: 'FREELANCER',
        provider: 'email',
        profile: {
          create: {
            firstName: 'Sarah',
            lastName: 'Freelancer',
            countryId: 'switzerland',
            phone: '+352 555 123 456',
            bio: 'Experienced web developer with 5+ years of experience in React, Node.js, and modern web technologies.',
            title: 'Full Stack Web Developer',
            hourlyRate: 45,
            skills: JSON.stringify(['React', 'Node.js', 'TypeScript', 'MongoDB', 'AWS']),
            categories: {
              create: [
                { categoryId: categories[0].id }, // Web Development
                { categoryId: categories[1].id }  // Mobile Development
              ]
            }
          }
        }
      }
    })
  ]);

  console.log('👥 Created sample users');

  // Create site settings
  await prisma.siteSettings.createMany({
    data: [
      {
        key: 'site_title',
        value: 'Smile24 - Freelancer Platform',
        type: 'text',
        description: 'Main site title'
      },
      {
        key: 'site_logo',
        value: '/assets/image/logo.png',
        type: 'text',
        description: 'Site logo URL'
      },
      {
        key: 'social_facebook',
        value: 'https://facebook.com/smile24',
        type: 'text',
        description: 'Facebook page URL'
      },
      {
        key: 'social_instagram',
        value: 'https://instagram.com/smile24',
        type: 'text',
        description: 'Instagram profile URL'
      },
      {
        key: 'social_linkedin',
        value: 'https://linkedin.com/company/smile24',
        type: 'text',
        description: 'LinkedIn company page URL'
      },
      {
        key: 'contact_email',
        value: 'support@smile24.com',
        type: 'text',
        description: 'Support email address'
      },
      {
        key: 'contact_phone',
        value: '+352 123 456 789',
        type: 'text',
        description: 'Support phone number'
      }
    ]
  });

  console.log('⚙️  Created site settings');

  console.log('✅ Database seeding completed successfully!');
  console.log('\n📋 Sample login credentials:');
  console.log('Admin: admin@smile24.com / admin@123');
  console.log('Client: client@example.com / admin@123');
  console.log('Freelancer: freelancer@example.com / admin@123');
}

main()
  .catch((e) => {
    console.error('❌ Seeding failed:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
