'use client';

import Link from 'next/link';
import { usePathname } from 'next/navigation';
import { useState, useEffect } from 'react';
import { useSession } from 'next-auth/react';
import { useProjectCounts } from '@/hooks/useProjectCounts';

interface FreelancerSidebarProps {
  locale: string;
  userProfile?: {
    name: string;
    profileImage: string;
    rating: number;
    reviewCount: number;
  };
  unreadMessagesCount?: number;
}

const FreelancerSidebar: React.FC<FreelancerSidebarProps> = ({ 
  locale, 
  userProfile,
  unreadMessagesCount = 0
}) => {
  const pathname = usePathname();
  const { data: session } = useSession();
  const [isProjectsOpen, setIsProjectsOpen] = useState(false);
  const [isProfileOpen, setIsProfileOpen] = useState(false);
  const { projectCounts } = useProjectCounts();

  // Determine if current page is related to projects or profile
  useEffect(() => {
    if (pathname.includes('/freelancer/projects') || pathname.includes('/freelancer/my-projects')) {
      setIsProjectsOpen(true);
    }
    if (pathname.includes('/freelancer/profile') || pathname.includes('/freelancer/on-board') || pathname.includes('/freelancer/add-experience')) {
      setIsProfileOpen(true);
    }
  }, [pathname]);



  const isActive = (path: string) => pathname.includes(path);
  const isProjectsActive = (type: string) => {
    if (typeof window !== 'undefined') {
      const urlParams = new URLSearchParams(window.location.search);
      const status = urlParams.get('status');
      return pathname.includes('/freelancer/my-projects') && status === type;
    }
    return false;
  };

  // Default profile data
  const defaultProfile = {
    name: session?.user?.name || userProfile?.name || 'Freelancer',
    profileImage: session?.user?.image || userProfile?.profileImage || '/assets/image/profile03.png',
    rating: userProfile?.rating || 0,
    reviewCount: userProfile?.reviewCount || 0
  };

  return (
    <div className="col-12 col-md-4 col-xl-3">
      <div className="customer-profile">
        <div className="freelancer-profile-edit">
          <div className="freelancer-profile-img">
            <img src={defaultProfile.profileImage} className="freelancer-profile-update-img" alt="Profile"
            onError={(e) => {
                e.currentTarget.onerror = null; // Prevent infinite loop
                e.currentTarget.src = "/assets/image/profile03.png";
              }}
            />
            <div className="edit-btn">
              <Link href={`/${locale}/freelancer/profile`} className="edit-btn-link" title="Edit Profile Image">
                <img src="/assets/image/edit-icon.svg" alt="Edit" width="20" height="20" />
              </Link>
            </div>
          </div>
          <h5>{defaultProfile.name}</h5>
          <span className="profile-reviews">
            <svg width="13" height="12" viewBox="0 0 13 12" fill="none" xmlns="http://www.w3.org/2000/svg">
              <path d="M10.2062 12L6.5 9.96162L2.79241 12L3.50213 7.67959L0.5 4.62053L4.64478 3.99059L6.5 0L8.35522 3.99059L12.5 4.62053L9.49787 7.68107L10.2062 12Z" fill="#FB6107"/>
            </svg>
            {/* {defaultProfile.rating} ({defaultProfile.reviewCount} Reviews) */}

            {
              (defaultProfile.rating || defaultProfile.reviewCount)
                ? `${defaultProfile.rating || 0} (${defaultProfile.reviewCount || 0} Reviews)`
                : 'No reviews yet'
            } 
            
          </span>
        </div>
      </div>
      
      <div className="customer-menu">
        <ul>
          {/* Projects Menu */}
          <li className={`dropdown-sub ${isProjectsOpen ? 'active' : ''} ${isActive('/freelancer/projects') || isActive('/freelancer/my-projects') ? 'page-active' : ''}`}>
            <a 
              href="#" 
              onClick={(e) => {
                e.preventDefault();
                setIsProjectsOpen(!isProjectsOpen);
              }}
            >
              <span>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M16.0002 22.7498H8.00023C3.38023 22.7498 2.52023 20.5998 2.30023 18.5098L1.55023 10.4998C1.44023 9.44977 1.41023 7.89977 2.45023 6.73977C3.35023 5.73977 4.84023 5.25977 7.00023 5.25977H17.0002C19.1702 5.25977 20.6602 5.74977 21.5502 6.73977C22.5902 7.89977 22.5602 9.44977 22.4502 10.5098L21.7002 18.4998C21.4802 20.5998 20.6202 22.7498 16.0002 22.7498ZM7.00023 6.74977C5.31023 6.74977 4.15023 7.07977 3.56023 7.73977C3.07023 8.27977 2.91023 9.10977 3.04023 10.3498L3.79023 18.3598C3.96023 19.9398 4.39023 21.2498 8.00023 21.2498H16.0002C19.6002 21.2498 20.0402 19.9398 20.2102 18.3498L20.9602 10.3598C21.0902 9.10977 20.9302 8.27977 20.4402 7.73977C19.8502 7.07977 18.6902 6.74977 17.0002 6.74977H7.00023Z" fill="#564B43"/>
                  <path d="M16 6.75C15.59 6.75 15.25 6.41 15.25 6V5.2C15.25 3.42 15.25 2.75 12.8 2.75H11.2C8.75 2.75 8.75 3.42 8.75 5.2V6C8.75 6.41 8.41 6.75 8 6.75C7.59 6.75 7.25 6.41 7.25 6V5.2C7.25 3.44 7.25 1.25 11.2 1.25H12.8C16.75 1.25 16.75 3.44 16.75 5.2V6C16.75 6.41 16.41 6.75 16 6.75Z" fill="#564B43"/>
                  <path d="M12 16.75C9.25 16.75 9.25 15.05 9.25 14.03V13C9.25 11.59 9.59 11.25 11 11.25H13C14.41 11.25 14.75 11.59 14.75 13V14C14.75 15.04 14.75 16.75 12 16.75ZM10.75 12.75C10.75 12.83 10.75 12.92 10.75 13V14.03C10.75 15.06 10.75 15.25 12 15.25C13.25 15.25 13.25 15.09 13.25 14.02V13C13.25 12.92 13.25 12.83 13.25 12.75C13.17 12.75 13.08 12.75 13 12.75H11C10.92 12.75 10.83 12.75 10.75 12.75Z" fill="#564B43"/>
                  <path d="M13.9995 14.7702C13.6295 14.7702 13.2995 14.4902 13.2595 14.1102C13.2095 13.7002 13.4995 13.3202 13.9095 13.2702C16.5495 12.9402 19.0795 11.9402 21.2095 10.3902C21.5395 10.1402 22.0095 10.2202 22.2595 10.5602C22.4995 10.8902 22.4295 11.3602 22.0895 11.6102C19.7495 13.3102 16.9895 14.4002 14.0895 14.7702C14.0595 14.7702 14.0295 14.7702 13.9995 14.7702Z" fill="#564B43"/>
                  <path d="M9.99975 14.7809C9.96975 14.7809 9.93975 14.7809 9.90975 14.7809C7.16975 14.4709 4.49975 13.4709 2.18975 11.8909C1.84975 11.6609 1.75975 11.1909 1.98975 10.8509C2.21975 10.5109 2.68975 10.4209 3.02975 10.6509C5.13975 12.0909 7.56975 13.0009 10.0697 13.2909C10.4797 13.3409 10.7797 13.7109 10.7297 14.1209C10.6997 14.5009 10.3797 14.7809 9.99975 14.7809Z" fill="#564B43"/>
                </svg>
              </span>
              Projects
              <span className={`arrow ${isProjectsOpen ? 'open' : ''}`}>
                <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M13.279 5.46727C13.1524 5.46727 13.0257 5.51393 12.9257 5.61393L8.57904 9.9606C8.25904 10.2806 7.73904 10.2806 7.41904 9.9606L3.07237 5.61393C2.87904 5.4206 2.55904 5.4206 2.3657 5.61393C2.17237 5.80727 2.17237 6.12726 2.3657 6.3206L6.71237 10.6673C7.41904 11.3739 8.57237 11.3739 9.2857 10.6673L13.6324 6.3206C13.8257 6.12726 13.8257 5.80727 13.6324 5.61393C13.5324 5.5206 13.4057 5.46727 13.279 5.46727Z" fill="#0E9493" stroke="#0E9493" strokeWidth="0.5"/>
                </svg>
              </span>
            </a>
            
            <ul className="submenus">
              <li className={`submenu-item ${isProjectsActive('recommended') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/my-projects?status=recommended`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M14.7161 19.3913C14.2744 19.3913 13.7078 19.2496 12.9994 18.8329L10.5078 17.3579C10.2494 17.2079 9.74945 17.2079 9.49945 17.3579L6.99945 18.8329C5.52445 19.7079 4.65778 19.3579 4.26611 19.0746C3.88278 18.7913 3.28278 18.0663 3.67445 16.3996L4.26611 13.8413C4.33278 13.5746 4.19945 13.1163 3.99945 12.9163L1.93278 10.8496C0.899447 9.81628 0.98278 8.93294 1.12445 8.49961C1.26611 8.06628 1.71611 7.29961 3.14945 7.05794L5.80778 6.61628C6.05778 6.57461 6.41611 6.30794 6.52445 6.08294L7.99945 3.14128C8.66611 1.79961 9.54111 1.59961 9.99945 1.59961C10.4578 1.59961 11.3328 1.79961 11.9994 3.14128L13.4661 6.07461C13.5828 6.29961 13.9411 6.56628 14.1911 6.60794L16.8494 7.04961C18.2911 7.29128 18.7411 8.05794 18.8744 8.49128C19.0078 8.92461 19.0911 9.80794 18.0661 10.8413L15.9994 12.9163C15.7994 13.1163 15.6744 13.5663 15.7328 13.8413L16.3244 16.3996C16.7078 18.0663 16.1161 18.7913 15.7328 19.0746C15.5244 19.2246 15.1911 19.3913 14.7161 19.3913ZM9.99945 15.9913C10.4078 15.9913 10.8161 16.0913 11.1411 16.2829L13.6328 17.7579C14.3578 18.1913 14.8161 18.1913 14.9911 18.0663C15.1661 17.9413 15.2911 17.4996 15.1078 16.6829L14.5161 14.1246C14.3578 13.4329 14.6161 12.5413 15.1161 12.0329L17.1828 9.96628C17.5911 9.55794 17.7744 9.15794 17.6911 8.88294C17.5994 8.60794 17.2161 8.38294 16.6494 8.29128L13.9911 7.84961C13.3494 7.74128 12.6494 7.22461 12.3578 6.64128L10.8911 3.70794C10.6244 3.17461 10.2911 2.85794 9.99945 2.85794C9.70778 2.85794 9.37445 3.17461 9.11611 3.70794L7.64111 6.64128C7.34945 7.22461 6.64945 7.74128 6.00778 7.84961L3.35778 8.29128C2.79111 8.38294 2.40778 8.60794 2.31611 8.88294C2.22445 9.15794 2.41611 9.56628 2.82445 9.96628L4.89111 12.0329C5.39111 12.5329 5.64945 13.4329 5.49111 14.1246L4.89945 16.6829C4.70778 17.5079 4.84111 17.9413 5.01611 18.0663C5.19111 18.1913 5.64111 18.1829 6.37445 17.7579L8.86611 16.2829C9.18278 16.0913 9.59111 15.9913 9.99945 15.9913Z" fill="#787676"/>
                    </svg>
                  </span>
                  Recommended ({projectCounts.recommended})
                </Link>
              </li>
              <li className={`submenu-item ${isProjectsActive('proposed') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/my-projects?status=proposed`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M12.4993 11.291H6.66602C6.32435 11.291 6.04102 11.0077 6.04102 10.666C6.04102 10.3243 6.32435 10.041 6.66602 10.041H12.4993C12.841 10.041 13.1243 10.3243 13.1243 10.666C13.1243 11.0077 12.841 11.291 12.4993 11.291Z" fill="#787676"/>
                      <path d="M10.316 14.625H6.66602C6.32435 14.625 6.04102 14.3417 6.04102 14C6.04102 13.6583 6.32435 13.375 6.66602 13.375H10.316C10.6577 13.375 10.941 13.6583 10.941 14C10.941 14.3417 10.6577 14.625 10.316 14.625Z" fill="#787676"/>
                      <path d="M11.666 6.12435H8.33268C7.53268 6.12435 6.04102 6.12435 6.04102 3.83268C6.04102 1.54102 7.53268 1.54102 8.33268 1.54102H11.666C12.466 1.54102 13.9577 1.54102 13.9577 3.83268C13.9577 4.63268 13.9577 6.12435 11.666 6.12435ZM8.33268 2.79102C7.50768 2.79102 7.29102 2.79102 7.29102 3.83268C7.29102 4.87435 7.50768 4.87435 8.33268 4.87435H11.666C12.7077 4.87435 12.7077 4.65768 12.7077 3.83268C12.7077 2.79102 12.491 2.79102 11.666 2.79102H8.33268Z" fill="#787676"/>
                      <path d="M12.5 19.4587H7.5C2.81667 19.4587 1.875 17.3087 1.875 13.8337V8.83368C1.875 5.03368 3.25 3.40868 6.63333 3.23368C6.96667 3.21702 7.275 3.47535 7.29167 3.82535C7.30833 4.17535 7.04167 4.45868 6.7 4.47535C4.33333 4.60868 3.125 5.31702 3.125 8.83368V13.8337C3.125 16.917 3.73333 18.2087 7.5 18.2087H12.5C16.2667 18.2087 16.875 16.917 16.875 13.8337V8.83368C16.875 5.31702 15.6667 4.60868 13.3 4.47535C12.9583 4.45868 12.6917 4.15868 12.7083 3.81702C12.725 3.47535 13.025 3.20868 13.3667 3.22535C16.75 3.40868 18.125 5.03368 18.125 8.82535V13.8253C18.125 17.3087 17.1833 19.4587 12.5 19.4587Z" fill="#787676"/>
                    </svg>
                  </span>
                  Proposed ({projectCounts.proposed})
                </Link>
              </li>
              <li className={`submenu-item ${isProjectsActive('awarded') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/my-projects?status=awarded`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path fillRule="evenodd" clipRule="evenodd" d="M14.3699 1.75C14.7014 1.75 15.0194 1.8817 15.2538 2.11612C15.488 2.3503 15.6197 2.66785 15.6199 2.99901C15.6215 3.99033 15.6246 7.27152 15.6246 9.25C15.6246 10.7843 14.641 12.0407 13.5351 12.8775C12.6389 13.5556 11.5616 14.0369 10.6246 14.1943V18H13.1246C13.4698 18 13.7496 18.2798 13.7496 18.625C13.7496 18.9702 13.4698 19.25 13.1246 19.25H6.87461C6.52943 19.25 6.24961 18.9702 6.24961 18.625C6.24961 18.2798 6.52943 18 6.87461 18H9.37461V14.1948C8.43696 14.038 7.35853 13.5563 6.46165 12.8774C5.35615 12.0406 4.37305 10.7842 4.37305 9.25C4.37305 8.65155 4.37432 7.3893 4.37559 6.12789C4.37686 4.86626 4.37812 3.60497 4.37812 3.0082V3.00625C4.37916 2.67541 4.51131 2.35847 4.74562 2.12489C4.97982 1.89142 5.29699 1.76028 5.62767 1.76016M9.99805 13C10.7029 13 11.824 12.6047 12.7809 11.8807C13.7364 11.1577 14.3746 10.2266 14.3746 9.25C14.3746 7.27223 14.3715 3.9917 14.3699 3.00099L14.3699 3L5.62885 3.01016C5.62885 3.60832 5.62685 4.86785 5.62559 6.12726L5.62559 6.12914C5.62432 7.39097 5.62305 8.65236 5.62305 9.25C5.62305 10.2267 6.26103 11.1578 7.21609 11.8808C8.17251 12.6048 9.29326 13 9.99805 13Z" fill="#787676"/>
                      <path fillRule="evenodd" clipRule="evenodd" d="M1.25 4.25C1.25 3.90482 1.52982 3.625 1.875 3.625H5C5.34518 3.625 5.625 3.90482 5.625 4.25C5.625 4.59518 5.34518 4.875 5 4.875H2.5C2.5 5.83677 2.79468 6.8076 3.27565 7.52251C3.75892 8.24085 4.37016 8.625 5 8.625C5.34518 8.625 5.625 8.90482 5.625 9.25C5.625 9.59518 5.34518 9.875 5 9.875C3.81538 9.875 2.86412 9.15017 2.23851 8.22026C1.6106 7.28693 1.25 6.07026 1.25 4.875V4.25ZM14.375 4.25C14.375 3.90482 14.6548 3.625 15 3.625H18.125C18.4702 3.625 18.75 3.90482 18.75 4.25V4.875C18.75 6.07026 18.3894 7.28693 17.7615 8.22026C17.1359 9.15017 16.1846 9.875 15 9.875C14.6548 9.875 14.375 9.59518 14.375 9.25C14.375 8.90482 14.6548 8.625 15 8.625C15.6298 8.625 16.2411 8.24085 16.7244 7.52251C17.2053 6.8076 17.5 5.83677 17.5 4.875H15C14.6548 4.875 14.375 4.59518 14.375 4.25Z" fill="#787676"/>
                    </svg>
                  </span>
                  Awarded ({projectCounts.awarded})
                </Link>
              </li>
              <li className={`submenu-item ${isProjectsActive('cancelled') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/my-projects?status=cancelled`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M9.99935 19.4577C5.05768 19.4577 1.04102 15.441 1.04102 10.4993C1.04102 5.55768 5.05768 1.54102 9.99935 1.54102C14.941 1.54102 18.9577 5.55768 18.9577 10.4993C18.9577 15.441 14.941 19.4577 9.99935 19.4577ZM9.99935 2.79102C5.74935 2.79102 2.29102 6.24935 2.29102 10.4993C2.29102 14.7493 5.74935 18.2077 9.99935 18.2077C14.2493 18.2077 17.7077 14.7493 17.7077 10.4993C17.7077 6.24935 14.2493 2.79102 9.99935 2.79102Z" fill="#787676"/>
                      <path d="M7.64245 13.4841C7.48411 13.4841 7.32578 13.4258 7.20078 13.3008C6.95911 13.0591 6.95911 12.6591 7.20078 12.4174L11.9174 7.70078C12.1591 7.45911 12.5591 7.45911 12.8008 7.70078C13.0424 7.94245 13.0424 8.34245 12.8008 8.58411L8.08411 13.3008C7.96745 13.4258 7.80078 13.4841 7.64245 13.4841Z" fill="#787676"/>
                      <path d="M12.3591 13.4841C12.2008 13.4841 12.0424 13.4258 11.9174 13.3008L7.20078 8.58411C6.95911 8.34245 6.95911 7.94245 7.20078 7.70078C7.44245 7.45911 7.84245 7.45911 8.08411 7.70078L12.8008 12.4174C13.0424 12.6591 13.0424 13.0591 12.8008 13.3008C12.6758 13.4258 12.5174 13.4841 12.3591 13.4841Z" fill="#787676"/>
                    </svg>
                  </span>
                  Cancelled ({projectCounts.cancelled})
                </Link>
              </li>
              <li className={`submenu-item ${isProjectsActive('completed') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/my-projects?status=completed`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M9.99935 19.4577C5.05768 19.4577 1.04102 15.441 1.04102 10.4993C1.04102 5.55768 5.05768 1.54102 9.99935 1.54102C14.941 1.54102 18.9577 5.55768 18.9577 10.4993C18.9577 15.441 14.941 19.4577 9.99935 19.4577ZM9.99935 2.79102C5.74935 2.79102 2.29102 6.24935 2.29102 10.4993C2.29102 14.7493 5.74935 18.2077 9.99935 18.2077C14.2493 18.2077 17.7077 14.7493 17.7077 10.4993C17.7077 6.24935 14.2493 2.79102 9.99935 2.79102Z" fill="#787676"/>
                      <path d="M8.81719 13.4827C8.65052 13.4827 8.49219 13.416 8.37552 13.2993L6.01719 10.941C5.77552 10.6993 5.77552 10.2993 6.01719 10.0577C6.25885 9.81602 6.65885 9.81602 6.90052 10.0577L8.81719 11.9743L13.1005 7.69102C13.3422 7.44935 13.7422 7.44935 13.9839 7.69102C14.2255 7.93268 14.2255 8.33268 13.9839 8.57435L9.25885 13.2993C9.14219 13.416 8.98385 13.4827 8.81719 13.4827Z" fill="#787676"/>
                    </svg>
                  </span>
                  Completed ({projectCounts.completed})
                </Link>
              </li>
            </ul>
          </li>
          
          {/* Messages Menu */}
          <li className={isActive('/freelancer/messages') ? 'active' : ''}>
            <Link href={`/${locale}/freelancer/messages`}>
              <span>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M12 22.81C11.31 22.81 10.66 22.46 10.2 21.85L8.7 19.85C8.67 19.81 8.55 19.76 8.5 19.75H8C3.83 19.75 1.25 18.62 1.25 13V8C1.25 3.58 3.58 1.25 8 1.25H16C20.42 1.25 22.75 3.58 22.75 8V13C22.75 17.42 20.42 19.75 16 19.75H15.5C15.42 19.75 15.35 19.79 15.3 19.85L13.8 21.85C13.34 22.46 12.69 22.81 12 22.81ZM8 2.75C4.42 2.75 2.75 4.42 2.75 8V13C2.75 17.52 4.3 18.25 8 18.25H8.5C9.01 18.25 9.59 18.54 9.9 18.95L11.4 20.95C11.75 21.41 12.25 21.41 12.6 20.95L14.1 18.95C14.43 18.51 14.95 18.25 15.5 18.25H16C19.58 18.25 21.25 16.58 21.25 13V8C21.25 4.42 19.58 2.75 16 2.75H8Z" fill="#564B43"/>
                  <path d="M12 12C11.44 12 11 11.55 11 11C11 10.45 11.45 10 12 10C12.55 10 13 10.45 13 11C13 11.55 12.56 12 12 12Z" fill="#564B43"/>
                  <path d="M16 12C15.44 12 15 11.55 15 11C15 10.45 15.45 10 16 10C16.55 10 17 10.45 17 11C17 11.55 16.56 12 16 12Z" fill="#564B43"/>
                  <path d="M8 12C7.44 12 7 11.55 7 11C7 10.45 7.45 10 8 10C8.55 10 9 10.45 9 11C9 11.55 8.56 12 8 12Z" fill="#564B43"/>
                </svg>
              </span>
              Messages
              {unreadMessagesCount > 0 && (
                <span className="massage-count">{unreadMessagesCount}</span>
              )}
            </Link>
          </li>
          
          {/* Profile Menu */}
          <li className={`dropdown-sub ${isProfileOpen ? 'active' : ''} ${isActive('/freelancer/profile') || isActive('/freelancer/on-board') || isActive('/freelancer/add-experience') ? 'page-active' : ''}`}>
            <a 
              href="#" 
              onClick={(e) => {
                e.preventDefault();
                setIsProfileOpen(!isProfileOpen);
              }}
            >
              <span>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M12 12.75C8.83 12.75 6.25 10.17 6.25 7C6.25 3.83 8.83 1.25 12 1.25C15.17 1.25 17.75 3.83 17.75 7C17.75 10.17 15.17 12.75 12 12.75ZM12 2.75C9.66 2.75 7.75 4.66 7.75 7C7.75 9.34 9.66 11.25 12 11.25C14.34 11.25 16.25 9.34 16.25 7C16.25 4.66 14.34 2.75 12 2.75Z" fill="#0E9493"/>
                  <path d="M15.8196 22.7494C15.4396 22.7494 15.0796 22.6094 14.8196 22.3494C14.5096 22.0394 14.3696 21.5894 14.4396 21.1194L14.6296 19.7694C14.6796 19.4194 14.8896 19.0094 15.1396 18.7494L18.6796 15.2094C20.0996 13.7894 21.3496 14.5994 21.9596 15.2094C22.4796 15.7294 22.7496 16.2894 22.7496 16.8494C22.7496 17.4194 22.4896 17.9494 21.9596 18.4794L18.4195 22.0194C18.1695 22.2694 17.7496 22.4794 17.3996 22.5294L16.0495 22.7194C15.9695 22.7394 15.8996 22.7494 15.8196 22.7494ZM20.3096 15.9194C20.1296 15.9194 19.9696 16.0394 19.7396 16.2694L16.1996 19.8094C16.1696 19.8394 16.1196 19.9394 16.1196 19.9794L15.9396 21.2294L17.1896 21.0494C17.2296 21.0394 17.3295 20.9894 17.3595 20.9594L20.8996 17.4194C21.0596 17.2594 21.2496 17.0294 21.2496 16.8494C21.2496 16.6994 21.1296 16.4894 20.8996 16.2694C20.6596 16.0294 20.4796 15.9194 20.3096 15.9194Z" fill="#0E9493"/>
                  <path d="M20.9206 19.2194C20.8506 19.2194 20.7806 19.2094 20.7206 19.1894C19.4006 18.8194 18.3506 17.7694 17.9806 16.4494C17.8706 16.0494 18.1006 15.6394 18.5006 15.5294C18.9006 15.4194 19.3106 15.6494 19.4206 16.0494C19.6506 16.8694 20.3006 17.5194 21.1206 17.7494C21.5206 17.8594 21.7506 18.2794 21.6406 18.6694C21.5506 18.9994 21.2506 19.2194 20.9206 19.2194Z" fill="#0E9493"/>
                  <path d="M3.41016 22.75C3.00016 22.75 2.66016 22.41 2.66016 22C2.66016 17.73 6.85018 14.25 12.0002 14.25C13.0902 14.25 14.1702 14.41 15.1802 14.71C15.5802 14.83 15.8002 15.25 15.6802 15.64C15.5602 16.04 15.1402 16.26 14.7502 16.14C13.8702 15.88 12.9502 15.74 12.0002 15.74C7.68018 15.74 4.16016 18.54 4.16016 21.99C4.16016 22.41 3.82016 22.75 3.41016 22.75Z" fill="#0E9493"/>
                </svg>
              </span>
              My Profile
              <span className={`arrow ${isProfileOpen ? 'open' : ''}`}>
                <svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M13.279 5.46727C13.1524 5.46727 13.0257 5.51393 12.9257 5.61393L8.57904 9.9606C8.25904 10.2806 7.73904 10.2806 7.41904 9.9606L3.07237 5.61393C2.87904 5.4206 2.55904 5.4206 2.3657 5.61393C2.17237 5.80727 2.17237 6.12726 2.3657 6.3206L6.71237 10.6673C7.41904 11.3739 8.57237 11.3739 9.2857 10.6673L13.6324 6.3206C13.8257 6.12726 13.8257 5.80727 13.6324 5.61393C13.5324 5.5206 13.4057 5.46727 13.279 5.46727Z" fill="#0E9493" stroke="#0E9493" strokeWidth="0.5"/>
                </svg>
              </span>
            </a>
            
            <ul className="submenus">
              <li className={`submenu-item ${isActive('/freelancer/profile') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/profile`}>
                  <span className="icon">
                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M12 12.75C8.83 12.75 6.25 10.17 6.25 7C6.25 3.83 8.83 1.25 12 1.25C15.17 1.25 17.75 3.83 17.75 7C17.75 10.17 15.17 12.75 12 12.75ZM12 2.75C9.66 2.75 7.75 4.66 7.75 7C7.75 9.34 9.66 11.25 12 11.25C14.34 11.25 16.25 9.34 16.25 7C16.25 4.66 14.34 2.75 12 2.75Z" fill="#0E9493"/>
                      <path d="M15.8196 22.7494C15.4396 22.7494 15.0796 22.6094 14.8196 22.3494C14.5096 22.0394 14.3696 21.5894 14.4396 21.1194L14.6296 19.7694C14.6796 19.4194 14.8896 19.0094 15.1396 18.7494L18.6796 15.2094C20.0996 13.7894 21.3496 14.5994 21.9596 15.2094C22.4796 15.7294 22.7496 16.2894 22.7496 16.8494C22.7496 17.4194 22.4896 17.9494 21.9596 18.4794L18.4195 22.0194C18.1695 22.2694 17.7496 22.4794 17.3996 22.5294L16.0495 22.7194C15.9695 22.7394 15.8996 22.7494 15.8196 22.7494ZM20.3096 15.9194C20.1296 15.9194 19.9696 16.0394 19.7396 16.2694L16.1996 19.8094C16.1696 19.8394 16.1196 19.9394 16.1196 19.9794L15.9396 21.2294L17.1896 21.0494C17.2296 21.0394 17.3295 20.9894 17.3595 20.9594L20.8996 17.4194C21.0596 17.2594 21.2496 17.0294 21.2496 16.8494C21.2496 16.6994 21.1296 16.4894 20.8996 16.2694C20.6596 16.0294 20.4796 15.9194 20.3096 15.9194Z" fill="#0E9493"/>
                      <path d="M20.9206 19.2194C20.8506 19.2194 20.7806 19.2094 20.7206 19.1894C19.4006 18.8194 18.3506 17.7694 17.9806 16.4494C17.8706 16.0494 18.1006 15.6394 18.5006 15.5294C18.9006 15.4194 19.3106 15.6494 19.4206 16.0494C19.6506 16.8694 20.3006 17.5194 21.1206 17.7494C21.5206 17.8594 21.7506 18.2794 21.6406 18.6694C21.5506 18.9994 21.2506 19.2194 20.9206 19.2194Z" fill="#0E9493"/>
                      <path d="M3.41016 22.75C3.00016 22.75 2.66016 22.41 2.66016 22C2.66016 17.73 6.85018 14.25 12.0002 14.25C13.0902 14.25 14.1702 14.41 15.1802 14.71C15.5802 14.83 15.8002 15.25 15.6802 15.64C15.5602 16.04 15.1402 16.26 14.7502 16.14C13.8702 15.88 12.9502 15.74 12.0002 15.74C7.68018 15.74 4.16016 18.54 4.16016 21.99C4.16016 22.41 3.82016 22.75 3.41016 22.75Z" fill="#0E9493"/>
                    </svg>
                  </span>
                  Edit Profile
                </Link>
              </li>
              <li className={`submenu-item ${isActive('/freelancer/on-board') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/on-board`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M10.1004 11.7747C10.0837 11.7747 10.0587 11.7747 10.0421 11.7747C10.0171 11.7747 9.98372 11.7747 9.95872 11.7747C8.06706 11.7164 6.65039 10.2414 6.65039 8.42474C6.65039 6.57474 8.15872 5.06641 10.0087 5.06641C11.8587 5.06641 13.3671 6.57474 13.3671 8.42474C13.3587 10.2497 11.9337 11.7164 10.1254 11.7747C10.1087 11.7747 10.1087 11.7747 10.1004 11.7747ZM10.0004 6.30807C8.83372 6.30807 7.89206 7.25807 7.89206 8.41641C7.89206 9.55807 8.78372 10.4831 9.91706 10.5247C9.94206 10.5164 10.0254 10.5164 10.1087 10.5247C11.2254 10.4664 12.1004 9.54974 12.1087 8.41641C12.1087 7.25807 11.1671 6.30807 10.0004 6.30807Z" fill="#787676"/>
                      <path d="M10 19.4586C7.75836 19.4586 5.61669 18.6253 3.95836 17.1086C3.80836 16.9753 3.74169 16.7753 3.75836 16.5836C3.86669 15.5919 4.48336 14.6669 5.50836 13.9836C7.99169 12.3336 12.0167 12.3336 14.4917 13.9836C15.5167 14.6753 16.1334 15.5919 16.2417 16.5836C16.2667 16.7836 16.1917 16.9753 16.0417 17.1086C14.3834 18.6253 12.2417 19.4586 10 19.4586ZM5.06669 16.4169C6.45003 17.5753 8.19169 18.2086 10 18.2086C11.8084 18.2086 13.55 17.5753 14.9334 16.4169C14.7834 15.9086 14.3834 15.4169 13.7917 15.0169C11.7417 13.6503 8.26669 13.6503 6.20003 15.0169C5.60836 15.4169 5.21669 15.9086 5.06669 16.4169Z" fill="#787676"/>
                      <path d="M9.99935 19.4577C5.05768 19.4577 1.04102 15.441 1.04102 10.4993C1.04102 5.55768 5.05768 1.54102 9.99935 1.54102C14.941 1.54102 18.9577 5.55768 18.9577 10.4993C18.9577 15.441 14.941 19.4577 9.99935 19.4577ZM9.99935 2.79102C5.74935 2.79102 2.29102 6.24935 2.29102 10.4993C2.29102 14.7493 5.74935 18.2077 9.99935 18.2077C14.2493 18.2077 17.7077 14.7493 17.7077 10.4993C17.7077 6.24935 14.2493 2.79102 9.99935 2.79102Z" fill="#787676"/>
                    </svg>
                  </span>
                  On Board
                </Link>
              </li>
              <li className={`submenu-item ${isActive('/freelancer/add-experience') ? 'active' : ''}`}>
                <Link href={`/${locale}/freelancer/add-experience`}>
                  <span className="icon">
                    <svg width="20" height="21" viewBox="0 0 20 21" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M9.55065 12.5661C7.91732 12.5661 6.58398 11.2328 6.58398 9.59948C6.58398 7.96614 7.91732 6.63281 9.55065 6.63281C11.184 6.63281 12.5173 7.96614 12.5173 9.59948C12.5173 11.2328 11.184 12.5661 9.55065 12.5661ZM9.55065 7.89115C8.60898 7.89115 7.83398 8.65782 7.83398 9.60782C7.83398 10.5578 8.60065 11.3245 9.55065 11.3245C10.5006 11.3245 11.2673 10.5578 11.2673 9.60782C11.2673 8.65782 10.5006 7.89115 9.55065 7.89115Z" fill="#787676"/>
                      <path d="M13.8746 17.9586C13.5329 17.9586 13.2496 17.6753 13.2496 17.3336C13.2496 15.7336 11.5913 14.4336 9.54961 14.4336C7.50794 14.4336 5.84961 15.7336 5.84961 17.3336C5.84961 17.6753 5.56628 17.9586 5.22461 17.9586C4.88294 17.9586 4.59961 17.6753 4.59961 17.3336C4.59961 15.0503 6.81628 13.1836 9.54961 13.1836C12.2829 13.1836 14.4996 15.042 14.4996 17.3336C14.4996 17.6753 14.2163 17.9586 13.8746 17.9586Z" fill="#787676"/>
                      <path d="M9.58268 19.4583C4.87435 19.4583 1.04102 15.625 1.04102 10.9167C1.04102 6.20833 4.87435 2.375 9.58268 2.375C10.741 2.375 11.8577 2.59999 12.9077 3.04166C13.2077 3.16666 13.3577 3.49998 13.2577 3.80831C13.166 4.08331 13.1243 4.375 13.1243 4.66667C13.1243 5.15833 13.2577 5.64165 13.516 6.05832C13.6494 6.29165 13.8244 6.49998 14.0244 6.67498C14.7494 7.33331 15.8243 7.54169 16.666 7.24169C16.9743 7.12502 17.3244 7.28334 17.4494 7.59168C17.8994 8.65001 18.1243 9.77501 18.1243 10.925C18.1243 15.625 14.291 19.4583 9.58268 19.4583ZM9.58268 3.625C5.56602 3.625 2.29102 6.89167 2.29102 10.9167C2.29102 14.9417 5.56602 18.2083 9.58268 18.2083C13.5993 18.2083 16.8743 14.9417 16.8743 10.9167C16.8743 10.1167 16.741 9.32499 16.491 8.56666C15.341 8.76666 14.0827 8.40835 13.1993 7.60002C12.9077 7.35002 12.6493 7.04167 12.4493 6.7C12.0827 6.1 11.8827 5.39167 11.8827 4.66667C11.8827 4.44167 11.8994 4.22502 11.941 4.00835C11.1827 3.75002 10.391 3.625 9.58268 3.625Z" fill="#787676"/>
                      <path d="M15.8333 8.62565C14.85 8.62565 13.9167 8.259 13.1917 7.60067C12.9 7.35067 12.6417 7.04232 12.4417 6.70065C12.075 6.10065 11.875 5.39232 11.875 4.66732C11.875 4.24232 11.9417 3.82566 12.075 3.42566C12.2583 2.85899 12.575 2.33399 13 1.90899C13.75 1.14232 14.7583 0.708984 15.8417 0.708984C16.975 0.708984 18.05 1.19233 18.7833 2.02567C19.4333 2.75067 19.8 3.68398 19.8 4.66732C19.8 4.98398 19.7583 5.30065 19.675 5.60065C19.5917 5.97565 19.4333 6.36733 19.2167 6.70899C18.7333 7.52566 17.9667 8.13398 17.0667 8.41732C16.6917 8.55898 16.275 8.62565 15.8333 8.62565ZM15.8333 1.95898C15.0917 1.95898 14.4 2.25063 13.8917 2.77563C13.6 3.07563 13.3917 3.41729 13.2667 3.80896C13.175 4.08396 13.1333 4.37565 13.1333 4.66732C13.1333 5.15898 13.2667 5.6423 13.525 6.05897C13.6583 6.2923 13.8333 6.50063 14.0333 6.67563C14.7583 7.33396 15.8333 7.54234 16.675 7.24234C17.3083 7.04234 17.825 6.62564 18.1583 6.06731C18.3083 5.82564 18.4083 5.56733 18.4667 5.309C18.525 5.09233 18.55 4.88398 18.55 4.66732C18.55 4.00065 18.3 3.359 17.85 2.859C17.3417 2.284 16.6083 1.95898 15.8333 1.95898Z" fill="#787676"/>
                      <path d="M17.0756 5.27539H14.584C14.2423 5.27539 13.959 4.99206 13.959 4.65039C13.959 4.30872 14.2423 4.02539 14.584 4.02539H17.0756C17.4173 4.02539 17.7007 4.30872 17.7007 4.65039C17.7007 4.99206 17.4257 5.27539 17.0756 5.27539Z" fill="#787676"/>
                      <path d="M15.834 6.55025C15.4923 6.55025 15.209 6.26692 15.209 5.92525V3.43359C15.209 3.09193 15.4923 2.80859 15.834 2.80859C16.1757 2.80859 16.459 3.09193 16.459 3.43359V5.92525C16.459 6.27525 16.1757 6.55025 15.834 6.55025Z" fill="#787676"/>
                    </svg>
                  </span>
                  Add Experience
                </Link>
              </li>
            </ul>
          </li>
          
          {/* Membership Menu */}
          <li className={isActive('/freelancer/membership') ? 'active' : ''}>
            <Link href={`/${locale}/freelancer/membership`}>
              <span>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <mask id="mask0_5001_26350" style={{maskType:"luminance"}} maskUnits="userSpaceOnUse" x="0" y="0" width="24" height="24">
                    <path d="M0 1.90735e-06H24V24H0V1.90735e-06Z" fill="white"/>
                  </mask>
                  <g mask="url(#mask0_5001_26350)">
                    <path d="M6.37105 23.7823C6.54108 23.8243 6.72084 23.7743 6.84469 23.6504L11.7666 18.7286L11.83 18.6504C11.9584 18.4564 11.9374 18.1925 11.7666 18.0215C11.5957 17.8507 11.3317 17.829 11.1377 17.9571L11.0595 18.0215L6.72262 22.3584L5.60055 18.7403C5.55168 18.5832 5.4285 18.46 5.27144 18.4112L1.65328 17.2891L4.75973 14.1827L4.82418 14.1045C4.95228 13.9105 4.93057 13.6465 4.75973 13.4756C4.58874 13.3047 4.3249 13.2837 4.13082 13.4121L4.05269 13.4756L0.361288 17.167C0.237446 17.2909 0.187443 17.4706 0.229452 17.6407C0.271441 17.8106 0.399224 17.9461 0.566366 17.9981L4.72262 19.2891L6.01363 23.4454L6.03707 23.5059C6.10015 23.643 6.22232 23.7455 6.37105 23.7823Z" fill="#564B43"/>
                    <path d="M17.205 23.6934C17.3247 23.7854 17.4811 23.819 17.6298 23.7823C17.7997 23.7402 17.9353 23.6125 17.9873 23.4453L19.2773 19.2891L23.4345 17.9981C23.6016 17.9461 23.7295 17.8105 23.7714 17.6407C23.8134 17.4706 23.7634 17.2909 23.6396 17.167L20.0888 13.6162C19.8936 13.421 19.5771 13.4212 19.3818 13.6162C19.1865 13.8115 19.1865 14.128 19.3818 14.3233L22.3466 17.2891L18.7295 18.4112C18.5723 18.46 18.4493 18.5832 18.4004 18.7403L17.2773 22.3584L12.9882 18.0684C12.793 17.8731 12.4765 17.8731 12.2812 18.0684C12.086 18.2636 12.086 18.5802 12.2812 18.7754L17.1562 23.6504L17.205 23.6934Z" fill="#564B43"/>
                    <path d="M20.2656 9.46875C20.2656 14.0205 16.5518 17.7344 12 17.7344C7.4482 17.7344 3.73438 14.0205 3.73438 9.46875C3.73438 4.91695 7.4482 1.20312 12 1.20312C16.5518 1.20312 20.2656 4.91695 20.2656 9.46875ZM21.2656 9.46875C21.2656 4.36467 17.1041 0.203125 12 0.203125C6.89592 0.203125 2.73438 4.36467 2.73438 9.46875C2.73438 14.5728 6.89592 18.7344 12 18.7344C17.1041 18.7344 21.2656 14.5728 21.2656 9.46875Z" fill="#564B43"/>
                    <path d="M17.5469 9.46875C17.5469 12.5231 15.0544 15.0156 12 15.0156C8.94563 15.0156 6.45312 12.5231 6.45312 9.46875C6.45312 6.41443 8.94562 3.92188 12 3.92188C15.0544 3.92188 17.5469 6.41443 17.5469 9.46875ZM18.5469 9.46875C18.5469 5.86214 15.6067 2.92188 12 2.92188C8.39334 2.92188 5.45312 5.86214 5.45312 9.46875C5.45312 13.0754 8.39334 16.0156 12 16.0156C15.6067 16.0156 18.5469 13.0754 18.5469 9.46875Z" fill="#564B43"/>
                    <path d="M13.8243 12.8085C13.9813 12.867 14.1574 12.8425 14.293 12.744C14.4285 12.6455 14.5052 12.4856 14.4981 12.3182L14.4151 10.3749L15.626 8.84852C15.7302 8.71726 15.7618 8.54209 15.71 8.3827C15.6582 8.22344 15.5296 8.10047 15.3682 8.05555L13.4913 7.53309L12.417 5.91102C12.3244 5.77136 12.1676 5.68739 12 5.68739C11.8325 5.68739 11.6757 5.77136 11.5831 5.91102L10.5079 7.53309L8.63188 8.05555C8.47052 8.10047 8.34193 8.22343 8.29009 8.3827C8.2383 8.54209 8.26987 8.71726 8.37407 8.84852L9.58403 10.3749L9.502 12.3182C9.49485 12.4856 9.57159 12.6455 9.70708 12.744C9.84266 12.8425 10.0188 12.867 10.1758 12.8085L12 12.1288L13.8243 12.8085ZM12.1749 11.1259C12.0622 11.0839 11.9379 11.0839 11.8252 11.1259L10.5323 11.6083L10.5918 10.2294C10.5969 10.1094 10.5581 9.99142 10.4834 9.89735L9.62602 8.81727L10.9551 8.44715L11.0381 8.4159C11.1185 8.37716 11.1874 8.31725 11.2374 8.24207L12 7.09266L12.7627 8.24207L12.8184 8.31141C12.8801 8.3757 12.9581 8.42295 13.045 8.44715L14.3731 8.81727L13.5166 9.89735C13.442 9.99143 13.4032 10.1094 13.4083 10.2294L13.4668 11.6083L12.1749 11.1259Z" fill="#564B43"/>
                  </g>
                </svg>
              </span>
              Membership
            </Link>
            <button className="add-membership-btn" onClick={() => {}}>
              <svg width="20" height="20" viewBox="0 0 20 20" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M6.66602 10H13.3327" stroke="#0E9493" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
                <path d="M10 13.3327V6.66602" stroke="#0E9493" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
                <path d="M7.49935 18.3327H12.4993C16.666 18.3327 18.3327 16.666 18.3327 12.4993V7.49935C18.3327 3.33268 16.666 1.66602 12.4993 1.66602H7.49935C3.33268 1.66602 1.66602 3.33268 1.66602 7.49935V12.4993C1.66602 16.666 3.33268 18.3327 7.49935 18.3327Z" stroke="#0E9493" strokeWidth="1.5" strokeLinecap="round" strokeLinejoin="round"/>
              </svg>
            </button>
          </li>
          
          {/* Logout Menu */}
          <li>
            <a href="/api/auth/signout">
              <span>
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                  <path d="M15.2395 22.2705H15.1095C10.6695 22.2705 8.52953 20.5205 8.15953 16.6005C8.11953 16.1905 8.41953 15.8205 8.83953 15.7805C9.23953 15.7405 9.61953 16.0505 9.65953 16.4605C9.94953 19.6005 11.4295 20.7705 15.1195 20.7705H15.2495C19.3195 20.7705 20.7595 19.3305 20.7595 15.2605V8.74047C20.7595 4.67047 19.3195 3.23047 15.2495 3.23047H15.1195C11.4095 3.23047 9.92953 4.42047 9.65953 7.62047C9.60953 8.03047 9.25953 8.34047 8.83953 8.30047C8.41953 8.27047 8.11953 7.90047 8.14953 7.49047C8.48953 3.51047 10.6395 1.73047 15.1095 1.73047H15.2395C20.1495 1.73047 22.2495 3.83047 22.2495 8.74047V15.2605C22.2495 20.1705 20.1495 22.2705 15.2395 22.2705Z" fill="#F24747"/>
                  <path d="M15.0001 12.75H3.62012C3.21012 12.75 2.87012 12.41 2.87012 12C2.87012 11.59 3.21012 11.25 3.62012 11.25H15.0001C15.4101 11.25 15.7501 11.59 15.7501 12C15.7501 12.41 15.4101 12.75 15.0001 12.75Z" fill="#F24747"/>
                  <path d="M5.85043 16.0998C5.66043 16.0998 5.47043 16.0298 5.32043 15.8798L1.97043 12.5298C1.68043 12.2398 1.68043 11.7598 1.97043 11.4698L5.32043 8.11984C5.61043 7.82984 6.09043 7.82984 6.38043 8.11984C6.67043 8.40984 6.67043 8.88984 6.38043 9.17984L3.56043 11.9998L6.38043 14.8198C6.67043 15.1098 6.67043 15.5898 6.38043 15.8798C6.24043 16.0298 6.04043 16.0998 5.85043 16.0998Z" fill="#F24747"/>
                </svg>
              </span>
              Logout
            </a>
          </li>
        </ul>
      </div>
      

    </div>
  );
};

export default FreelancerSidebar;