'use client';

import { useState } from 'react';
import Image from 'next/image';
import Link from 'next/link';
import SimpleTextEditor from '@/components/ui/SimpleTextEditor';
import ReviewsList from '@/components/reviews/ReviewsList';
import '@/components/ui/SimpleTextEditor.css';

interface Document {
  id: string;
  filename: string;
  size: string;
  type: string;
  url: string;
}

interface WorkExperience {
  id: string;
  title: string;
  location: string;
  description: string;
  documents: Document[];
}

interface FreelancerProfile {
  id: string;
  name: string;
  email: string;
  profileImage: string;
  bio: string;
  title: string;
  rating: number;
  reviewCount: number;
  phone?: string;
  location: string;
  completedProjects: number;
  memberSince: string;
  workExperiences: WorkExperience[];
}

interface FreelancerProfileViewProps {
  freelancer: FreelancerProfile;
  locale: string;
}

export default function FreelancerProfileView({ freelancer, locale }: FreelancerProfileViewProps) {
  const [editableBio, setEditableBio] = useState(freelancer.bio);
  const [isOwnerView, setIsOwnerView] = useState(false); // This would be determined by checking if current user owns this profile

  const handleDocumentDownload = (document: Document) => {
    // In a real application, this would handle the actual download
    console.log('Downloading document:', document.filename);
    // For now, we'll just open the URL
    if (document.url) {
      window.open(document.url, '_blank');
    }
  };

  const handleBioChange = (newBio: string) => {
    setEditableBio(newBio);
    // In a real application, you would save this to the backend
    console.log('Bio updated:', newBio);
  };

  const renderStarRating = (rating: number) => {
    return (
      <svg width="13" height="12" viewBox="0 0 13 12" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path 
          d="M10.2062 12L6.5 9.96162L2.79241 12L3.50213 7.67959L0.5 4.62053L4.64478 3.99059L6.5 0L8.35522 3.99059L12.5 4.62053L9.49787 7.68107L10.2062 12Z" 
          fill="#FB6107"
        />
      </svg>
    );
  };

  const renderDocumentIcon = (type: string) => {
    // Default PNG icon - in a real app, you'd have different icons for different file types
    return (
      <Image
        src="/assets/image/png.svg"
        alt="Document"
        width={24}
        height={24}
      />
    );
  };

  const renderDownloadIcon = () => (
    <svg width="19" height="18" viewBox="0 0 19 18" fill="none" xmlns="http://www.w3.org/2000/svg">
      <path 
        d="M10.0294 16.9298C9.89508 17.0642 9.71123 17.149 9.4991 17.149C9.08898 17.149 8.74957 16.8096 8.74957 16.3995V-1.39949C8.74957 -1.80962 9.08898 -2.14903 9.4991 -2.14903C9.90922 -2.14903 10.2486 -1.80962 10.2486 -1.39949L10.2486 16.3995C10.2486 16.6116 10.1638 16.7955 10.0294 16.9298Z" 
        fill="#EF7310"
      />
      <path 
        d="M17.2912 9.66705L10.0292 16.929C9.73933 17.2189 9.2585 17.2189 8.96859 16.929L1.7066 9.66705C1.41669 9.37714 1.41669 8.8963 1.7066 8.60639C1.99651 8.31647 2.47735 8.31647 2.76726 8.60639L9.49892 15.338L16.2306 8.60639C16.5205 8.31648 17.0013 8.31648 17.2912 8.60639C17.5811 8.8963 17.5811 9.37714 17.2912 9.66705Z" 
        fill="#EF7310"
      />
    </svg>
  );

  return (
    <section className="profile-wrapper create-project-wrapper py-60 freelancer-profile-view">
      <div className="container">
        <div className="row">
          <div className="col-12">
            <div className="back-button">
              <Link href={`/${locale}/freelancers`} className="back-link">
                <img
                  src="/assets/image/back-arrow.svg"
                  alt="Back to freelancers"
                />
                Back to Freelancers
              </Link>
            </div>
            <div className="page-title">
              <h2>{freelancer.name}&apos;s Profile</h2>
            </div>
          </div>
          
          <div className="col-12 col-md-4 col-xl-3">
            <div className="customer-profile">
              <div className="freelancer-profile-view-edit-block">
                <div className="freelancer-profile-view-img">
                  <Image
                    src={freelancer.profileImage}
                    className="freelancer-profile-view-update-img"
                    alt={freelancer.name}
                    width={100}
                    height={100}

                      onError={(e) => {
                        e.currentTarget.onerror = null; // Prevent infinite loop
                        e.currentTarget.src = "/assets/image/profile03.png";
                      }}
                  />
                </div>
                <h5>{freelancer.name}</h5>
                <span className="profile-reviews">
                  {renderStarRating(freelancer.rating)}
                  {freelancer.rating} ({freelancer.reviewCount}+ Reviews)
                </span>
                <div className="customer-position">
                  {freelancer.title}
                </div>
                <div className="divider"></div>
                <ul className="fc-contact">
                  <li>
                    <a href={`mailto:${freelancer.email}`} className="fc-text">
                      <div className="icon">
                        <Image
                          src="/assets/image/gmail.svg"
                          alt="Email"
                          width={16}
                          height={16}
                        />
                      </div>
                      {freelancer.email}
                    </a>
                  </li>
                  {freelancer.phone && (
                    <li>
                      <a href={`tel:${freelancer.phone}`} className="fc-text">
                        <div className="icon">
                          <Image
                            src="/assets/image/call.svg"
                            alt="Phone"
                            width={16}
                            height={16}
                          />
                        </div>
                        {freelancer.phone}
                      </a>
                    </li>
                  )}
                  <li className="fc-text">
                    <div className="icon">
                      <Image
                        src="/assets/image/location.svg"
                        alt="Location"
                        width={16}
                        height={16}
                      />
                    </div>
                    {freelancer.location}
                  </li>
                  <li className="fc-text">
                    <div className="icon">
                      <Image
                        src="/assets/image/project-comp.svg"
                        alt="Projects"
                        width={16}
                        height={16}
                      />
                    </div>
                    {freelancer.completedProjects} Projects Completed
                  </li>
                  <li className="fc-text">
                    <div className="icon">
                      <Image
                        src="/assets/image/timer-prim.svg"
                        alt="Member since"
                        width={16}
                        height={16}
                      />
                    </div>
                    Member Since {freelancer.memberSince}
                  </li>
                </ul>
              </div>
            </div>
          </div>
          
          <div className="col-12 col-md-8 col-xl-9">
            <div className="profile-about mt-0 ">
              <div className="about-header">
                <h3>About Me</h3>
                {/* Demo toggle - remove in production */}
                {/* <button 
                  type="button"
                  onClick={() => setIsOwnerView(!isOwnerView)}
                  className="demo-toggle"
                  title={isOwnerView ? "Switch to public view" : "Switch to owner view (demo)"}
                >
                  {isOwnerView ? "👁️ Public View" : "✏️ Edit Mode"}
                </button> */}
                {/* <button type="button"
                  onClick={() => setIsOwnerView(!isOwnerView)}
                  className="demo-toggle flex items-center gap-2"
                  title={isOwnerView ? "Switch to public view" : "Switch to owner view (demo)"}
                >
                  {isOwnerView ? (
                    <>
                      <svg width="18" height="18" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg">
                      <path d="M0.75 9C0.75 9 3.75 3 9 3C14.25 3 17.25 9 17.25 9C17.25 9 14.25 15 9 15C3.75 15 0.75 9 0.75 9Z" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"/>
                      <path d="M9 11.25C10.2426 11.25 11.25 10.2426 11.25 9C11.25 7.75736 10.2426 6.75 9 6.75C7.75736 6.75 6.75 7.75736 6.75 9C6.75 10.2426 7.75736 11.25 9 11.25Z" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"/>
                      </svg>
                      <span>Public View</span>
                    </>
                  ) : (
                    <>
                      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path><path d="m18.5 2.5 3 3L12 15l-4 1 1-4 9.5-9.5z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path></svg>
                      <span>Edit Mode</span>
                    </>
                  )}
                </button> */}
              </div>
              <SimpleTextEditor
                value={editableBio}
                onChange={handleBioChange}
                placeholder="Click to add your bio and tell clients about your experience, skills, and what makes you unique..."
                disabled={!isOwnerView}
                className="bio-editor"
              />
              
              {freelancer.workExperiences.length > 0 && (
                <>
                  <h3>Work Experience</h3>
                  <div className="work-exp-outer">
                    {freelancer.workExperiences.map((experience) => (
                      <div key={experience.id} className="work-exp-card">
                        <h4>{experience.title}</h4>
                        <div className="work-location">{experience.location}</div>
                        <div className="work-dis">
                          <p>{experience.description}</p>
                        </div>
                        
                        {experience.documents.length > 0 && (
                          <div className="doc-grid">
                            {experience.documents.map((document) => (
                              <div 
                                key={document.id} 
                                className="doc-card img"
                                onClick={() => handleDocumentDownload(document)}
                              >
                                <div className="doc-left">
                                  <div className="icon-box">
                                    {renderDocumentIcon(document.type)}
                                  </div>
                                  <div className="doc-text">
                                    <div className="filename">{document.filename}</div>
                                    <div className="size">{document.size}</div>
                                  </div>
                                </div>
                                <div className="download-arrow">
                                  {renderDownloadIcon()}
                                </div>
                              </div>
                            ))}
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                </>
              )}

              {/* Reviews Section */}
              <div className="reviews-section mt-4">
                <h3>Client Reviews</h3>
                <ReviewsList 
                  freelancerId={freelancer.id} 
                  showAll={false} 
                  limit={5} 
                />
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}